<?php

namespace App\Http\Requests\Api\V100;

use App\Helpers\Constant;
use App\Services\RespondActive;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Http\Exceptions\HttpResponseException;
use Illuminate\Validation\Rule;

class GetFilterRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        return [
            'category_id'       => [
                'nullable',
                Rule::exists('categories', 'id')
                    ->where('active', Constant::CATEGORY_STATUS['Active'])
                    ->whereNull('parent_id')
            ],
            'sub_category_id'   => [
                'nullable',
                Rule::exists('categories', 'id')
                    ->where('active', Constant::CATEGORY_STATUS['Active'])
                    ->where('parent_id', request()->category_id)
            ],
            'properties'        => ['nullable', 'array', 'min:1'],
            'properties.*.id'   => ['nullable', Rule::exists('properties', 'id')],
            'properties.*.from' => ['nullable', 'numeric', 'min:0'],
            'properties.*.to'   => ['nullable', 'numeric'],
            'country_id'        => [
                'required',
                Rule::exists('countries', 'id')
                    ->where('active', Constant::COUNTRY_STATUS['Active'])
            ],
            'city_ids'          => ['required', 'array', 'min:1'],
            'city_ids.*'        => [
                'required',
                Rule::exists('countries', 'id')
                    ->where('active', Constant::COUNTRY_STATUS['Active'])
                    ->where('parent_id', request()->country_id)
            ],
            'min_price'         => ['nullable', 'numeric', 'min:0'],
            'max_price'         => ['nullable', 'numeric'],
            'order_by'          => [
                'required',
                Rule::in([
                    Constant::FILTER_ORDER_BY['Newest'],
                    Constant::FILTER_ORDER_BY['Least price'],
                    Constant::FILTER_ORDER_BY['Highest price'],
                ])
            ],
            'advertisement_type' => [
                'required',
                Rule::in([
                    Constant::FILTER_ADVERTISEMENT_TYPE['All'],
                    Constant::FILTER_ADVERTISEMENT_TYPE['Premium'],
                    Constant::FILTER_ADVERTISEMENT_TYPE['Free'],
                ])
            ],
            'rates'             => ['required', 'array'],
            'rates.*'           => ['required', 'min:1', 'max:5']
        ];
    }

    protected function failedValidation(Validator $validator)
    {
        throw new HttpResponseException(RespondActive::clientError(
            RespondActive::stringifyErrors($validator->errors())
        ));
    }
}
