<?php

namespace App\Models;

use App\Helpers\Constant;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable;

    protected $fillable = [
        'account_type',
        'verified',
        'name',
        'nickname',
        'phone',
        'country_code',
        'email',
        'city_id',
        'latitude',
        'longitude',
        'address',
        'premium',
        'social_media',
        'social_media_objective',
        'social_media_provider_id',
        'average_rate',
        'rate_count',
        'platform',
        'image',
        'background_image',
        'social_media_links',
        'working_hours',
        'description',
        'views_count'
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
    ];

    public function image()
    {
        $this->load('hubFiles');

        return $this->hubFiles?->get_path();
    }

    protected function socialMediaLinks(): Attribute
    {
        return Attribute::make(
            get: fn ($value) => json_decode($value),
            set: fn ($value) => json_encode($value),
        );
    }

    protected function workingHours(): Attribute
    {
        return Attribute::make(
            get: fn ($value) => json_decode($value),
            set: fn ($value) => json_encode($value),
        );
    }

    public function scopeCheckUserExist($query, $phone)
    {
        return $query->where(['phone' => $phone, 'account_type' => Constant::USER_TYPE['User']]);
    }

    public function scopeCheckStoreExist($query, $phone)
    {
        return $query->where(['phone' => $phone, 'account_type' => Constant::USER_TYPE['Store']]);
    }

    public function city()
    {
        return $this->belongsTo(Country::class, 'city_id');
    }

    public function advertisements()
    {
        return $this->hasMany(Advertisement::class,'store_id','id');
    }

    public function explores()
    {
        return $this->hasMany(Explore::class,'store_id','id');
    }

    public function exploreActions($type)
    {
        return $this->belongsToMany(Explore::class)
            ->wherePivot('type', $type)
            ->withTimestamps();
    }

    public function scopeVerified($query)
    {
        $query->where('verified', Constant::USER_STATUS['Verified'])->whereNull('deleted_at');
    }

    public function scopeCheckUserType($query, $type)
    {
        $query->where('account_type', $type);
    }

    public function scopeUserAdvertisementCount($query)
    {
        $query->withCount('advertisements', 'followers')->withSum('advertisements', 'views_count');;
    }

    public static function getVerifiedUserBasedOnType($type)
    {
        switch ($type) {
            case Constant::USER_TYPE['User']:
                return self::verified()->checkUserType(Constant::USER_TYPE['User'])->userAdvertisementCount();

            case Constant::USER_TYPE['Store']:
                return self::verified()->checkUserType(Constant::USER_TYPE['Store'])->userAdvertisementCount();
        }
    }

    public static function getVerifiedUserBasedOnTypeAndCountry($type, $country_id)
    {
        $cities = Country::find($country_id)->getActiveCities()->pluck('id');

        switch ($type) {
            case Constant::USER_TYPE['User']:
                return self::verified()->checkUserType(Constant::USER_TYPE['User'])->whereIn('city_id', $cities);

            case Constant::USER_TYPE['Store']:
                return self::verified()->checkUserType(Constant::USER_TYPE['Store'])->whereIn('city_id', $cities);
        }
    }

    public function followings()
    {
        return $this->belongsToMany(User::class, 'follower_user', 'follower_id', 'store_id')
            ->withTimestamps();
    }

    public function followers()
    {
        return $this->belongsToMany(User::class, 'follower_user', 'store_id', 'follower_id')
            ->withTimestamps();
    }
    public function categories()
    {
        return $this->belongsToMany(Category::class,'category_store','store_id')->withTimestamps();
    }

    public function hubFiles()
    {
        return $this->morphOne(HubFile::class, 'morphable');
    }

    public function advertisementsActions()
    {
        return $this->belongsToMany(Advertisement::class)->withTimestamps()->withPivot('type');
    }

    public function advertisementsActionsByType($type)
    {
        return $this->belongsToMany(Advertisement::class)
            ->wherePivot('type', $type)
            ->withTimestamps();
    }

    public function reports()
    {
        return $this->belongsToMany(User::class, 'user_store', 'user_id', 'store_id')
            ->wherePivot('type', Constant::USER_STORE_TYPE['Report'])
            ->withTimestamps();
    }

    public function blocks()
    {
        return $this->belongsToMany(User::class, 'user_store', 'user_id', 'store_id')
            ->wherePivot('type', Constant::USER_STORE_TYPE['Block'])
            ->withTimestamps();
    }

    public function stories()
    {
        return $this->hasMany(Story::class, 'store_id');
    }

    public function storyViews()
    {
        return $this->belongsToMany(Story::class, 'story_views');
    }

    public function packages()
    {
        return $this->belongsToMany(Package::class, 'package_store');
    }
    public function newStories()
    {
        return $this->hasMany(Story::class, 'store_id')->where('expired_at', '>', now());
    }
    public function stores()
    {
        return $this->belongsToMany(User::class, 'user_store', 'user_id', 'store_id')
            ->withPivot('type', 'reason_id')
            ->withTimestamps();
    }

    public function storesByType($type)
    {
        return $this->belongsToMany(User::class, 'user_store', 'user_id', 'store_id')
            ->wherePivot('type', $type)
            ->withPivot('type', 'reason_id')
            ->withTimestamps();
    }
    public function notifications()
    {
        return $this->hasMany(Notification::class);
    }
    public function reviews()
    {
        return $this->morphMany(Review::class,'reviewable');
    }
    public function views()
    {
        return $this->belongsToMany(User::class, 'user_store', 'store_id', 'user_id')
            ->wherePivot('type', Constant::USER_STORE_TYPE['View']);
    }
    public function blockedStores()
    {
        return $this->belongsToMany(User::class, 'user_store', 'user_id', 'store_id')
            ->wherePivot('type', Constant::USER_STORE_TYPE['Block']);
    }

    public function usersBlockedStores()
    {
        return $this->belongsToMany(User::class, 'user_store', 'store_id', 'user_id')
            ->wherePivot('type', Constant::USER_STORE_TYPE['Block']);
    }

}
