<?php

namespace App\Services\Advertisement;

use App\Helpers\Constant;
use App\Http\Resources\Advertisement\AdvertisementResource;
use App\Http\Resources\Category\CategoryResource;
use App\Models\Advertisement;
use App\Models\Category;

class AdvertisementService
{
    /**
     * Return the users with type store based on parameters.
     *
     * @param  Array  $data ['search', 'store_id', 'country_id', 'city_ids', 'category_id', 'sub_category_id', 'properties',
     *                       'usersActionsType', 'premium', 'random', 'max_price', 'min_price', 'followings',
     *                       'max_rate', 'min_rate', 'orderByKey', 'orderBySort', 'limit', 'paginate']
     */
    public static function advertisementsBasedOnParameters($data)
    {
        $advertisements = Advertisement::with( ['city.country', 'hubFiles', 'users','subCategory'=>function($query){
            $query->with('category');
        },'shop'])
            ->when($data['search'] ?? false, function ($query) use ($data) {
                $query->where('title', 'like', '%' . $data['search'] . '%');
            })
            ->when(isset($data['store_id']) && $data['store_id'] != 'All', function ($query) use($data) {
                $query->where('store_id', $data['store_id']);
            })
            ->when(isset($data['country_id']) && $data['country_id'] != 'All', function ($query) use($data) {
                $query->whereHas('city.country', function ($query) use ($data) {
                    $query->where('id', $data['country_id']);
                });
            })
            ->when($data['city_id'] ?? false, function ($query) use ($data) {
                $query->where('city_id', $data['city_id']);
            })
            ->when($data['city_ids'] ?? false, function ($query) use ($data) {
                $query->whereIn('city_id', $data['city_ids']);
            })
            ->when($data['category_id'] ?? false, function ($query) use($data) {
                $query->whereHas('subCategory.category', function ($query) use ($data) {
                    $query->where('id', $data['category_id']);
                });
            })
            ->when($data['sub_category_id'] ?? false, function ($query) use($data) {
                $query->where('sub_category_id', $data['sub_category_id']);
            })
            ->when($data['property_id'] ?? false, function ($query) use ($data) {
                $query->whereHas('properties', function ($query) use ($data) {
                    $query->where('property_id', $data['property_id']);
                });
            })
            ->when($data['properties'] ?? false, function ($query) use ($data) {
                $query->whereHas('properties', function ($query) use ($data) {
                    foreach ($data['properties'] as $property) {
                        $query->where('advertisement_property.property_id', $property['id'])
                            ->when($property['from'] ?? false && $property['to'] ?? false, function ($query) use($property){
                                $query->whereBetween('advertisement_property.text', [$property['from'], $property['to']]);

                            });
                    }
                });
            })
            ->when($data['usersActionsType'] ?? false, function ($query) use($data) {
                $query->whereHas('users', function ($query) use ($data) {
                    $query->where(['type'=> $data['usersActionsType'],'user_id'=>auth()->id() ]);
                });
            })
            ->when(isset($data['checkPremium']) &&  $data['checkPremium'] != 'All'? true : false , function ($query) use($data) {
                $query->where('premium', $data['checkPremium']);
            })
            ->when($data['premium'] ?? false == Constant::PREMIUM_STATUS['True'], function ($query) use($data) {
                $query->premium(Constant::PREMIUM_STATUS['True']);
            })
            ->when(isset($data['other_options']) && $data['other_options'] != Constant::FILTER_ADVERTISEMENT_TYPE['All'], function ($query) use($data) {
                $premium = $data['other_options'] == Constant::FILTER_ADVERTISEMENT_TYPE['Premium'] ?
                    Constant::PREMIUM_STATUS['True'] : Constant::PREMIUM_STATUS['False'];
                $query->premium($premium);
            })
            ->when($data['random'] ?? false, function ($query) use($data) {
                $query->inRandomOrder();
            })
            ->when($data['max_price'] ?? false && $data['min_price'] ?? false, function ($query) use ($data) {
                $query->where('price', '>=', $data['min_price'] )->where('price', '<=', $data['max_price'] );
            })
            ->when($data['max_rate'] ?? false && $data['min_rate'] ?? false, function ($query) use ($data) {
                $query->where('average_rate', '>=', $data['min_rate'] )->where('average_rate', '<=', $data['max_rate'] );
            })
            ->when(isset($data['type']) && $data['type'] == Constant::MOST_POPULAR_ADS['Suggested ads'], function ($query) use($data) {
                $query->orderBy('id', 'desc');
            })
            ->when(isset($data['type']) && $data['type'] == Constant::MOST_POPULAR_ADS['Most popular ads'], function ($query) use($data) {
                $query->orderBy('views_count', 'desc');
            })
            ->when($data['orderByKey'] ?? false && $data['orderBySort'] ?? false, function ($query) use($data) {
                $query->orderBy($data['orderByKey'], $data['orderBySort']);
            });

        if ($data['paginate'] ?? false) {
            return $advertisements->paginate($data['limit']);
        }

        return $advertisements->take($data['limit'])->get();
    }

    /**
     * Return the users with type store based om paramters.
     *
     * @param  Array  $data ['category_id', 'premium', 'followings', 'most_rated', 'limit']
     */
    public static function advertisementsBasedOnCategory($data)
    {
        return AdvertisementResource::collection(
            Category::with('advertisements')
            ->find($data['category_id'])
            ->advertisements()
            ->with('store', 'city.country')
            ->premium(Constant::PREMIUM_STATUS['True'])
            ->paginate($data['limit'])
        );
    }

    /**
     * Return the users with type store based om paramters.
     *
     * @param  Array  $data ['category_id', 'premium', 'followings', 'most_rated', 'limit']
     */
    public static function advertisementsBasedOnCategoryInHomePage()
    {
        return CategoryResource::collection(
            Category::with(['categoryAdvertisements' => fn ($query) => $query->where('premium', true),'translation'])
            ->where('app_home', Constant::CATEGORY_APP_HOME['True'])
            ->get()
        );
    }
}
